#ifndef CE2_RND_H
#define CE2_RND_H

/*
 * All the includes that are needed for code using this module to
 * compile correctly should be #included here.
 */
#include "CE2_public.h"

#ifdef __cplusplus
extern "C"
{
#endif

  /*
   *  Object CE2_RND.h
   *  State           :  %state%
   *  Creation date   :  Thu Dec 16 17:11:32 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief This file contains the CE2 APIs used for random numbergeneration.
   *
   *  \version CE2_RND.h#1:incl:15
   *  \author adams
   *  \remarks Copyright (C) 2005 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************************ Defines ******************************/

/** 
@brief - a definition describing the low level Engine type ( SW , Hardware , Etc )
*/

#define CE2_RND_ENGINE_TYPE  LLF_RND_ENGINE_TYPE

/************************ Enums ********************************/
/************************ Typedefs  ****************************/
/************************ Structs  ******************************/
/************************ Public Variables **********************/
/************************ Public Functions **********************/

#ifdef RND_FIPS_PUB_186_2

/**
****************************************************************
* Function Name: 
*  CE2_RND_GenerateVector
*
*  @param[in] RndSize - The size of random vector that is required.
*  @param[in/out] Output_ptr - The output vector.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  The CE2_RND_GenerateVector function generates a random vector using
*  the FIPS-PUB 186-2 standard, Appendix 3.
*  The random function is based on the HASH SHA1 and works as follows:
*  1. Loads the secret key stored internally into a global variable.
*  2. Executes the following steps according to the number of hash result 
*     blocks required (RndSize/20).
*    a. Executes a SHA1 hash on the current key
*       (if the key is less than 512 bytes, adds zeros to it).
*    b. Stores the result to the data 
*    c. The next key is 1 ^ key ^ result. This is the new key.
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RND_GenerateVector; 
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RND_GenerateVector(DxUint16_t  RndSize,
                                            DxUint8_t  *Output_ptr);

/**
****************************************************************
* Function Name: 
*  CE2_RND_ResetSeed
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  The CE2_RND_ResetSeed resets the SEED generated by the low level
*  (the hardware or operating system service on software).
*
*  \b 
* Algorithm:
*  -# Call low level function LLF_RND_ResetSeed; 
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RND_ResetSeed(void);

/**
****************************************************************
* Function Name: 
*  CE2_RND_GenerateVectorInRange
*
*  @RndSize[in] - The size of random vectore that is required.
*  @MaxVect_ptr[in]  - The pointer to vector defines high limit of random vector.
*  @RndVect_ptr[in/out] The output vector.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  This function generates a random vector RndVect in range:
*  1 < RndVect < MaxVect,   using the FIPS-PUB 186-2 standard appendex 3 :
*  The function performs the following:
*    1. Calls the CE2_RND_GenerateVector() function for generating random vector 
*       RndVect of size in bytes.
*    2. Zeroes extra bits of RndVect.
*    3. Compares RndVect with 1 and MaxVect. If condition  1 < RandVect < MaxVect
*       is not satisfied goues to step 1.
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function CE2_RND_GenerateVectorInRange; 
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RND_GenerateVectorInRange(DxUint32_t RndSizeInBits, 
                                                   DxUint8_t *MaxVect_ptr, 
                                                   DxUint8_t *RndVect_ptr);
											  
#elif RND_FIPS_SP_800_90

/**
****************************************************************
* Function Name: 
*  CE2_RND_Instantiation
*
* Input:
*  None
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  Initializes RND engine according to FIPS-SP 800-90:
*  - CCSystemSpec_RNG v1.2 [2.3.2.2 Instantiation];
*  - RNG_SW_SDDv0.3.
*
*  \b 
* Algorithm:
*  -# Call low level function LLF_RND_Instantiation;
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RND_Instantiation(void);

/**
****************************************************************
* Function Name: 
*  CE2_RND_GenerateVector
*
* Input:
*  @param[in]     RndSize    - The size of random vector that is required;
*  @param[in/out] Output_ptr - The output vector.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  Generates pseudo random data vector according to FIPS-SP 800-90:
*  - CCSystemSpec_RNG v1.2 [2.3.2.4 Random Number Generation];
*  - RNG_SW_SDDv0.3.
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RND_GenerateVector;
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RND_GenerateVector(DxUint16_t RndSize,		/* in */
											   DxUint8_t  *Output_ptr);	/* in/out */

/**
****************************************************************
* Function Name: 
*  CE2_RND_Reseeding
*
* Input:
*  None
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  Preforms RND engine reseeding according to FIPS-SP 800-90:
*  - CCSystemSpec_RNG v1.2 [2.3.2.3 Reseeding];
*  - RNG_SW_SDDv0.3.
*
*  \b 
* Algorithm:
*  -# Call low level function LLF_RND_Reseeding;
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RND_Reseeding(void);

/**
****************************************************************
* Function Name: 
*  CE2_RND_AddAdditionalInput
*
* Input:
*  @param[in] AdditionalInput     - Additional input data;
*  @param[in] AdditionalInputSize - Additional input data size.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  Adds additional info vector to RND engine according to:
*  - CCSystemSpec_RNG v1.2;
*  - RNG_SW_SDDv0.3.
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RND_AddAdditionalInput;
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RND_AddAdditionalInput(DxUint8_t  *AdditionalInput,		/* in */
												   DxUint16_t AdditionalInputSize);	/* in */

#endif

#ifdef __cplusplus
}
#endif

#endif